//--------------------------------------------------------------------------------------
// File:		Font.cpp.
// Namespace:	Global.
// Description:	A wrapper for a FreeType font.
// Author:		Grant Davies.
// Platform:	ALL.
// 
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Includes.
//--------------------------------------------------------------------------------------

#ifndef __FONT__
#include "Font.h"
#endif //__FONT__

#include <string>
#include <iostream>

#include <ft2build.h>
#include FT_FREETYPE_H

#ifndef __CORE__
#include "Core.h"
#endif //__CORE__

#ifndef __TYPECONVERSIONS__
#include "TypeConversions.h"
#endif //__TYPECONVERSIONS__


//--------------------------------------------------------------------------------------
// Constants.
//--------------------------------------------------------------------------------------

#define CLASS_NAME Font


//--------------------------------------------------------------------------------------
// Function Definitions.
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Description:	Constructor.
// Parameters:	The free type library,
//				The name of the font face to load.
// Returns:		None.
//--------------------------------------------------------------------------------------
Font::Font(const FT_Library& freeTypeLibrary, std::string strFontFace) :
	pointSize(0),
	xOffsetPadding(0)
{
	// Load the type face.
	self.lastErrorValue = FT_New_Face(freeTypeLibrary, strFontFace.c_str(), 0, &self.typeFace);
}

//--------------------------------------------------------------------------------------
// Description:	Set the point size of this font.
// Parameters:	The point size of this font.
// Returns:		true if the size was set successfully; false if not.
//--------------------------------------------------------------------------------------
bool Font::setSize(float size)
{
	// Set the char size.
	float fontWidthPt = size;
	float fontHeightPt = 0;
	int deviceWidthDpi = 0;
	int deviceHeightDpi = 0;

	self.lastErrorValue = 
		FT_Set_Char_Size(
			self.typeFace,
			fontWidthPt * 64,
			fontHeightPt * 64,
			deviceWidthDpi,
			deviceHeightDpi
		);

	self.pointSize = size;

	return self.isStatusOk();
}

//--------------------------------------------------------------------------------------
// Description:	Get the point size of this font.
// Parameters:	None.
// Returns:		The point size of this font.
//--------------------------------------------------------------------------------------
int Font::getSize() const
{
	return self.pointSize;
}

//--------------------------------------------------------------------------------------
// Description:	Load a glyph from this font.
// Parameters:	The index of the glyph to load (not the same as the character code),
//				[optional] The type of fix to apply for the x offset issue. (default = 
//				XOffsetFixType_None),
// Returns:		A pointer to the loaded glyph object, or 0 if the glyph could not be
//				loaded.
//--------------------------------------------------------------------------------------
Glyph* Font::loadGlyph(int glyphIndex, Glyph::XOffsetFixType xOffsetFixType)
{
	// Load the glyph.
	Glyph* glyph = new Glyph(self.typeFace, glyphIndex, xOffsetFixType, self.xOffsetPadding);

	if (glyph->isStatusError())
	{
		// Get the error value.
		self.lastErrorValue = glyph->getLastErrorValue();

		// Delete the glyph.
		deleteAndClear(glyph);
	}

	return glyph;
}

//--------------------------------------------------------------------------------------
// Description:	Translate a character code to a glyph index for this font.
// Parameters:	The ASCII character code to translate.
// Returns:		The glyph index corresponding to the specified ASCII code.
//--------------------------------------------------------------------------------------
int Font::getGlyphIndexForCharacterCode(char characterCode) const
{
	return FT_Get_Char_Index(self.typeFace, characterCode);
}

//--------------------------------------------------------------------------------------
// Description:	Get the title of this font.
// Parameters:	None.
// Returns:		The title of this font.
//--------------------------------------------------------------------------------------
std::string Font::getTitle() const
{
	std::string title = typeFace->family_name;

	// Replace spaces with underscores.
	for (int i = 0; i < title.size(); i++)
	{
		if (' ' == title[i])
		{
			title[i] = '_';
		}
	}

	// Truncate at 20 chars.
	if (int(title.size()) > 20)
	{
		title = title.substr(0, 20);
	}

	return title;
}

//--------------------------------------------------------------------------------------
// Description:	Check the x offsets of the glyphs in this font, and calculate the global
//				x offset padding value.
// Parameters:	None.
// Returns:		true if offsets are okay; false if offsets are not okay.
//--------------------------------------------------------------------------------------
bool Font::checkXOffsets()
{
	bool okay = true;

	// Count the number of characters with an x offset for informational purposes.
	self.numCharactersWithNegativeOffset = 0;
	self.numCharactersWithPositiveOffset = 0;

	// Reset the x offset add.
	self.xOffsetPadding = 0;

	// Export all characters.
	for (int characterCode = 0; characterCode < numCharactersToExport; characterCode++)
	{
		// Get the index of the glyph that represents this character.
		int glyphIndex = getGlyphIndexForCharacterCode(characterCode);

		// Load the glyph.
		Glyph* glyph = loadGlyph(glyphIndex, Glyph::XOffsetFixType_None);
		if (glyph)
		{
			// Check the bitmap left padding.
			int bitmapLeft = glyph->getBitmapLeft();
			if (bitmapLeft > 0)
			{
				self.numCharactersWithPositiveOffset++;
			}
			if (bitmapLeft < 0)
			{
				// Extra space needs to be added to each glyph for the font to render
				// correctly.
				int xAdd = std::abs(bitmapLeft);
				if (xAdd > self.xOffsetPadding)
				{
					self.xOffsetPadding = xAdd;
				}

				// Count the number of characters with a negative x offset for
				// informational purposes.
				self.numCharactersWithNegativeOffset++;
			}
		}
		else
		{
			// Output warning about missing character code.
			std::cerr 
				<< "Warning: unable to load character code " 
				<< getCharacterCodeString(characterCode) 
				<< std::endl;

			okay = false;
		}
	}

	return okay;
}

//--------------------------------------------------------------------------------------
// Description:	Get the number of characters in this font with a negative x offset.
// Parameters:	None.
// Returns:		The number of characters in this font with a negative x offset.
//--------------------------------------------------------------------------------------
int Font::getNumCharactersWithNegativeXOffset() const
{
	return self.numCharactersWithNegativeOffset;
}

//--------------------------------------------------------------------------------------
// Description:	Get the number of characters in this font with a positive x offset.
// Parameters:	None.
// Returns:		The number of characters in this font with a positive x offset.
//--------------------------------------------------------------------------------------
int Font::getNumCharactersWithPositiveXOffset() const
{
	return self.numCharactersWithPositiveOffset;
}

//--------------------------------------------------------------------------------------
// Description:	Get the amount of padding that will be added to the left of each glyph
//				in this font.
// Parameters:	None.
// Returns:		The amount of padding that will be added to the left of each glyph.
//--------------------------------------------------------------------------------------
int Font::getXOffsetPadding() const
{
	return self.xOffsetPadding;
}
