//--------------------------------------------------------------------------------------
// File:		FontEngine.cpp.
// Namespace:	Global.
// Description:	A wrapper for the FreeType font engine.
// Author:		Grant Davies.
// Platform:	ALL.
// 
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Includes.
//--------------------------------------------------------------------------------------

#ifndef __FONTENGINE__
#include "FontEngine.h"
#endif //__FONTENGINE__

#include <string>

#include <ft2build.h>
#include FT_FREETYPE_H

#ifndef __CORE__
#include "Core.h"
#endif //__CORE__


//--------------------------------------------------------------------------------------
// Constants.
//--------------------------------------------------------------------------------------

#define CLASS_NAME FontEngine


//--------------------------------------------------------------------------------------
// Function Definitions.
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Description:	Constructor.
// Parameters:	None.
// Returns:		None.
//--------------------------------------------------------------------------------------
FontEngine::FontEngine() :
	initialised(false)
{
}

//--------------------------------------------------------------------------------------
// Description:	Initialise this font engine.
// Parameters:	None.
// Returns:		true if this font engine was initialised successfully; false if not.
//--------------------------------------------------------------------------------------
bool FontEngine::initialise()
{
	bool statusOkay = true;

	if (!self.initialised)
	{
		// Initialise the FreeType library.
		self.lastErrorValue = FT_Init_FreeType(&self.freeTypeLibrary);

		statusOkay = isStatusOk();

		if (statusOkay)
		{
			self.initialised = true;
		}
	}

	// Return whether the library was successfully initialised (true) or not (false).
	return statusOkay;
}

//--------------------------------------------------------------------------------------
// Description:	Load the specified font.
// Parameters:	The name of the font to load.
// Returns:		A pointer to the loaded font, or 0 if the font could not be loaded.
//--------------------------------------------------------------------------------------
Font* FontEngine::loadFont(std::string strFontFace)
{
	// Initialise first if not already initialised.
	if (!self.initialised)
	{
		// Not already initialised; initialise the font engine.
		self.initialise();
	}

	Font* font = 0;

	// Only attempt to load the font if this font engine has been initialised.
	if (self.initialised)
	{
		// Load the specified font.
		font = new Font(self.freeTypeLibrary, strFontFace);

		// Get the last error value.
		self.lastErrorValue = font->getLastErrorValue();

		// Delete the font if there is an error.
		if (font->isStatusError())
		{
			// Delete the font.
			deleteAndClear(font);
		}
	}

	// Return the loaded font (or 0 if the font could not be loaded).
	return font;
}
