//--------------------------------------------------------------------------------------
// File:		FontExporter.cpp.
// Namespace:	Global.
// Description:	The font exporter application.
// Author:		Grant Davies.
// Platform:	ALL.
// 
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Includes.
//--------------------------------------------------------------------------------------

#ifndef __FONTEXPORTER__
#include "FontExporter.h"
#endif //__FONTEXPORTER__

#include <assert.h>
#include <iostream>
#include <fstream>
#include <algorithm>
#include <cctype>
#include <direct.h>			// mkdir().

#ifndef __CORE__
#include "Core.h"
#endif //__CORE__

#ifndef __GLYPHPAGE__
#include "GlyphPage.h"
#endif //__GLYPHPAGE__

#ifndef __DESCRIPTORFILE__
#include "DescriptorFile.h"
#endif //__DESCRIPTORFILE__

#ifndef __DOOM3GLYPHDESCRIPTOR__
#include "Doom3GlyphDescriptor.h"
#endif //__DOOM3GLYPHDESCRIPTOR__

#ifndef __TYPECONVERSIONS__
#include "TypeConversions.h"
#endif //__TYPECONVERSIONS__


//--------------------------------------------------------------------------------------
// Constants.
//--------------------------------------------------------------------------------------

#define CLASS_NAME FontExporter

const std::string toolName = "ExportFontToDoom3";
const std::string toolVersion = "v1.02";

//--------------------------------------------------------------------------------------
// Function Definitions.
//--------------------------------------------------------------------------------------


//--------------------------------------------------------------------------------------
// Description:	Constructor.
// Parameters:	None.
// Returns:		None.
//--------------------------------------------------------------------------------------
FontExporter::FontExporter() :
	font(0),
	xOffsetFixType(Glyph::XOffsetFixType_Font),
	noXOffsetWarnings(false)
{
	std::cout << toolName << " " << toolVersion << "." << std::endl;
	std::cout << "Grant Davies 2005." << std::endl;
}

//--------------------------------------------------------------------------------------
// Description:	Destructor.
// Parameters:	None.
// Returns:		None.
//--------------------------------------------------------------------------------------
FontExporter::~FontExporter()
{
	// Destroy the font.
	deleteAndClear(self.font);
}

//--------------------------------------------------------------------------------------
// Description:	Initialise this exporter application.
// Parameters:	None.
// Returns:		true if this exporter initialised successfully; false if not.
//--------------------------------------------------------------------------------------
bool FontExporter::initialise(int argumentCount, char* arguments[])
{
	// Parse the arguments.
	bool parsed = parseArguments(argumentCount, arguments);
	if (!parsed)
	{
		return false;
	}

	// Initialise the image library.
	ilInit();

	// Initialise the font engine.
	bool okay = self.fontEngine.initialise();
	if (!okay)
	{
		std::cerr
			<< "Error: Unable to load the FreeType font engine."
			<< std::endl;

		return false;
	}

	// Load the font.
	self.font = self.fontEngine.loadFont(self.fontFileName);
	if (!self.font)
	{
		// Error code means that the font file could not 
		// be opened or read, or simply that it is broken.
		std::cerr
			<< "Error: Unable to load font " << self.fontFileName
			<< std::endl;

		return false;
	}

	// Set the font title.
	if (self.fontTitle.empty())
	{
		self.fontTitle = self.font->getTitle();
	}

	// Initialised successfully; return true.
	return true;
}

//--------------------------------------------------------------------------------------
// Description:	Convert the specified string to all lowercase.
// Parameters:	The string to convert.
// Returns:		The converted string.
//--------------------------------------------------------------------------------------
static inline std::string toLower(std::string s)
{
	std::transform(s.begin(), s.end(), s.begin(), std::tolower);
	return s;
}

//--------------------------------------------------------------------------------------
// Description:	Parse the command-line arguments.
// Parameters:	The number of arguments,
//				The array of arguments.
// Returns:		None.
//--------------------------------------------------------------------------------------
bool FontExporter::parseArguments(int argumentCount, char* arguments[])
{
	bool okay = true;

	// Parse the arguments.
	for (int argumentIndex = 1; argumentIndex < argumentCount; argumentIndex++)
	{
		// Parse the current argument.
		std::string argument = arguments[argumentIndex];

		// Determine whether it is a switch argument.
		char switchChar = argument.at(0);
		if ('-' == switchChar || '/' == switchChar)
		{
			// Switch argument.
			argument = toLower(argument.substr(1, argument.size() - 1));

			// Check for "size" switch.
			if (toLower("size") == argument)
			{
				// "size" specification.
				// Expected: size integer argument.
				argumentIndex++;
				if (argumentIndex < argumentCount)
				{
					argument = arguments[argumentIndex];

					int fontPointSize = toInteger(argument);

					self.fontPointSizes.push_back(fontPointSize);
				}
				else
				{
					// Size argument missing value.
					std::cerr
						<< "Error: no size specified after -size argument.  Integer expected."
						<< std::endl;
				}
			}
			// Check for "xOffsetFix" switch.
			else if (toLower("xOffsetFix") == argument)
			{
				// Expected: fix type Enum argument.
				argumentIndex++;
				if (argumentIndex < argumentCount)
				{
					// Get the fix type.
					std::string fixType = toLower(arguments[argumentIndex]);

					if ("none" == fixType)
					{
						self.xOffsetFixType = Glyph::XOffsetFixType_None;
					}
					else if ("glyph" == fixType)
					{
						self.xOffsetFixType = Glyph::XOffsetFixType_Glyph;
					}
					else if ("font" == fixType)
					{
						self.xOffsetFixType = Glyph::XOffsetFixType_Font;
					}
					else
					{
						// Unrecognised fix type.
						std::cerr
							<< "Error: unrecognised fix type specified after -xOffsetFix argument.  Expected: none, glyph, or font."
							<< std::endl;
					}
				}
				else
				{
					// Fix type argument missing value.
					std::cerr
						<< "Error: no fix type specified after -xOffsetFix argument.  Expected: none, glyph, or font."
						<< std::endl;
				}
			}
			// Check for the "textureFormat" switch.
			else if (toLower("textureFormat") == argument)
			{
				// Expected: format string argument.
				argumentIndex++;
				if (argumentIndex < argumentCount)
				{
					// Get the texture format.
					std::string textureFormat = toLower(arguments[argumentIndex]);

					// Warn if unrecognised.
					if (textureFormat != "tga" && textureFormat != "dds")
					{
						std::cerr
							<< "Warning: non-standard or unrecognised texture format \""
							<< textureFormat
							<< "\". Expected formats are \"tga\" and \"dds\"."
							<< std::endl;
					}

					// Add the texture format.
					self.textureFormats.push_back(textureFormat);
				}
				else
				{
					// Texture format argument missing value.
					std::cerr
						<< "Error: no format specified after -textureFormat argument.  String expected."
						<< std::endl;
				}
			}
			// Check for "noXOffsetWarnings" switch.
			else if (toLower("noXOffsetWarnings") == argument)
			{
				// No parameters associated with this argument.
				self.noXOffsetWarnings = true;
			}
			// Check for "name" switch.
			else if (toLower("name") == argument)
			{
				// Expected: name string argument.
				argumentIndex++;
				if (argumentIndex < argumentCount)
				{
					// Get the name.
					self.fontTitle = toLower(arguments[argumentIndex]);
				}
				else
				{
					// Name argument missing value.
					std::cerr
						<< "Error: no name specified after -name argument.  String expected."
						<< std::endl;
				}
			}
			// Check for "?" switch.
			else if (toLower("?") == argument)
			{
				outputUsage();
				return false;
			}
		}
		else
		{
			// Not a switch argument.

			// Must be the font file name.
			if (self.fontFileName.empty())
			{
				self.fontFileName = argument;
			}
			else
			{
				// Already have the font file name; error.
				std::cerr
					<< "Error: Unknown argument - \"" << argument << "\"."
					<< std::endl;

				okay = false;

				break;
			}
		}
	}

	// Check for the font file name.
	if (self.fontFileName.empty())
	{
		// Not enough command line arguments.
		std::cerr
			<< "Error: No font file name specified."
			<< std::endl;

		okay = false;
	}

	if (!okay)
	{
		// Output the usage.
		outputUsage();

		return false;
	}

	// Add default sizes if none have been specified.
	if (self.fontPointSizes.empty())
	{
		self.fontPointSizes.push_back(12);
		self.fontPointSizes.push_back(24);
		self.fontPointSizes.push_back(48);
	}

	// Add default texture format if none have been specified.
	if (self.textureFormats.empty())
	{
		self.textureFormats.push_back("tga");
	}

	// Parsed okay; return true.
	return true;
}

//--------------------------------------------------------------------------------------
// Description:	Output the usage of this application.
// Parameters:	None.
// Returns:		None.
//--------------------------------------------------------------------------------------
void FontExporter::outputUsage() const
{
	std::cerr
		<< "Usage: " << toolName << " "
		<< "<font file name> "
		<< "[-size <point size>] "
		<< "[-xOffsetFix <fix type>] "
		<< "[-textureFormat <format>] "
		<< "[-noXOffsetWarnings] "
		<< "[-name <name>] "
		<< std::endl;
}

//--------------------------------------------------------------------------------------
// Description:	Export the current font at the requested sizes.
// Parameters:	None.
// Returns:		None.
//--------------------------------------------------------------------------------------
bool FontExporter::export()
{
	bool okay = true;

	// Sort the sizes.
	std::sort(self.fontPointSizes.begin(), self.fontPointSizes.end());

	// Get the largest size.
	int nLargestSize = self.fontPointSizes.back();

	// Iterate through the sizes, exporting each one.
	for (IntVector::iterator fontSizeIterator = self.fontPointSizes.begin();
		fontSizeIterator != self.fontPointSizes.end() && okay;
		++fontSizeIterator)
	{
		int fontPointSize = (*fontSizeIterator);
		float glyphScale = float(nLargestSize) / float(fontPointSize);

		okay = export(fontPointSize, glyphScale);
	}

	return okay;
}

//--------------------------------------------------------------------------------------
// Description:	Export the current font at the specified size.
// Parameters:	The size of the font to export,
//				The scale of the glyph to export.
// Returns:		None.
//--------------------------------------------------------------------------------------
bool FontExporter::export(int fontSize, float glyphScale)
{
	std::cout << "Exporting " << self.fontTitle << " " << fontSize << "pt." << std::endl;

	// Clear out old data.
	self.glyphIndicesToCharacterCodes.clear();
	self.glyphPages.clear();

	// Set the font size.
	bool okay = self.font->setSize(fontSize);
	if (!okay)
	{
		std::cerr
			<< "Error: Unable to set the font size."
			<< std::endl;

		return false;
	}

	// Check the characters and set up the x offset.
	okay = checkCharacters();
	if (!okay)
	{
		std::cerr
			<< "Error: Unable to export font."
			<< std::endl;

		return false;
	}

	// Create the output folder.
	self.outputFolderName = self.fontTitle;
	::mkdir(self.outputFolderName.c_str());

	// Calculate the font descriptor file title.
	std::string fontDescriptorFileTitle = "fontImage_" + toString(fontSize);

	// Open the glyph descriptor output file.
	std::string descriptorFileName = self.outputFolderName + "/" + fontDescriptorFileTitle + ".dat";
	self.descriptorFile.open(descriptorFileName, DescriptorFile::Mode_Binary);

	// Export all characters.
	for (int characterCode = 0; characterCode < Font::numCharactersToExport; characterCode++)
	{
		bool okay = exportCharacter(characterCode);
		if (!okay)
		{
			std::cerr
				<< "Error: Unable to export character " 
				<< getCharacterCodeString(characterCode) << "."
				<< std::endl;

			return false;
		}
	}

	// Write the trailing header to the descriptor file.
	self.descriptorFile.writeTrailer(glyphScale, "fonts/" + fontDescriptorFileTitle);

	// Write all texture pages to image files.
	for (int pageIndex = 0; pageIndex < self.glyphPages.size(); pageIndex++)
	{
		// Iterate through the texture formats, outputting each one.
		for (std::vector<std::string>::iterator textureFormatIterator = self.textureFormats.begin();
			textureFormatIterator != self.textureFormats.end();
			++textureFormatIterator)
		{
			// Get the current texture format.
			std::string textureFormat = (*textureFormatIterator);

			// Write the current page to file.
			std::string fileName = 
				self.outputFolderName + 
				"/" + self.glyphPages[pageIndex].getFileName() + 
				"." + textureFormat;

			// Write this texture format to disk.
			self.glyphPages[pageIndex].writeToFile(fileName);

			// Output some information.
			std::cout << "Wrote texture page \"" << fileName << "\"." << std::endl;
		}
	}

	// Close the descriptor file.
	self.descriptorFile.close();

	// Export was successful.
	std::cout << "Font exported successfully." << std::endl;

	// Success; return true.
	return true;
}

//--------------------------------------------------------------------------------------
// Description:	Check all the characters in the current font to ensure that they can be
//				loaded and calculate the amount that needs to be added to each glyph to
//				account for the negative x offset issue.
// Parameters:	None.
// Returns:		true if font is okay; false if font is broken.
//--------------------------------------------------------------------------------------
bool FontExporter::checkCharacters()
{
	// Check the x offsets.
	bool okay = self.font->checkXOffsets();

	if (noXOffsetWarnings)
	{
		return okay;
	}

	int numCharactersWithPositiveOffset = self.font->getNumCharactersWithPositiveXOffset();
	int numCharactersWithNegativeOffset = self.font->getNumCharactersWithNegativeXOffset();

	int numCharactersWithXOffset = 
		numCharactersWithPositiveOffset + numCharactersWithNegativeOffset;

	if (numCharactersWithXOffset > 0)
	{
		// Output a warning if no fix is being applied.
		if (Glyph::XOffsetFixType_None == self.xOffsetFixType)
		{
			// No fix being applied; output a warning to the user.
			std::cerr 
				<< "Warning: " 
				<< numCharactersWithXOffset
				<< " out of "
				<< 256
				<< " characters have a modified x offset. "
				<< "Export with the \"-xOffsetFixType\" switch to fix this font or "
				<< "the glyphs that use this feature. The default Doom 3 font renderer "
				<< "will render this exported data very poorly. "
				<< "Export will continue without fixing the font."
				<< std::endl;
		}
		else if (Glyph::XOffsetFixType_Glyph == self.xOffsetFixType &&
			numCharactersWithNegativeOffset > 0)
		{
			// Glyph fix being applied; output a warning to the user.
			std::cerr 
				<< "Warning: " 
				<< numCharactersWithNegativeOffset 
				<< " out of "
				<< 256
				<< " characters have a negative x offset. "
				<< "Export with the \"-xOffsetFixType font\" switch to automatically "
				<< "account for this (font will then be offset on the x axis). "
				<< "This operation will continue to export without fixing the font "
				<< "(only the glyphs will be fixed that make use of the negative x "
				<< "offset, though they still will not render perfectly)."
				<< std::endl;
		}
		else
		{
			// Output information about how far this font will be offset.
			std::cout
				<< "This font will be offset by "
				<< self.font->getXOffsetPadding()
				<< " pixels."
				<< std::endl;
		}
	}

	return okay;
}

//--------------------------------------------------------------------------------------
// Description:	Export a single character.
// Parameters:	The ASCII character code to export.
// Returns:		true if the character was exported successfully; false if not.
//--------------------------------------------------------------------------------------
bool FontExporter::exportCharacter(int characterCode)
{
	Doom3GlyphDescriptor* doom3GlyphDescriptor = 0;

	// Get the index of the glyph that represents this character.
	int glyphIndex = self.font->getGlyphIndexForCharacterCode(characterCode);

	// Check whether this glyph has been already used.
	std::map<int, int>::iterator glyphIndexIterator = 
		self.glyphIndicesToCharacterCodes.find(glyphIndex);

	if (self.glyphIndicesToCharacterCodes.end() == glyphIndexIterator)
	{
		// Add a glyph index to character code mapping.
		self.glyphIndicesToCharacterCodes[glyphIndex] = characterCode;

		// Glyph index has not already been used; load the new glyph.
		Glyph* glyph = self.font->loadGlyph(glyphIndex, self.xOffsetFixType);
		if (glyph)
		{
			// Find the page to add the current glyph to.
			GlyphPage* targetGlyphPage;

			// Attempt to add the glyph to each page.
			bool canAddToPage = false;
			int pageIndex;
			for (pageIndex = 0; pageIndex < self.glyphPages.size(); pageIndex++)
			{
				// Add the glyph to the current page.
				canAddToPage = self.glyphPages[pageIndex].canAccomodateGlyph(*glyph);
				if (canAddToPage)
				{
					targetGlyphPage = &self.glyphPages[pageIndex];
					break;
				}
			}

			if (!canAddToPage)
			{
				// No room in any of the pages for the current glyph; create a
				// new glyph page and add it there.

				// Create the name of the new page.
				int pageIndex = self.glyphPages.size();
				std::string textureFileTitle = 
					self.fontTitle + "_" + 
					toString(pageIndex) + "_" + 
					toString(font->getSize());

				// Create the page.
				GlyphPage glyphPage;
				self.glyphPages.push_back(glyphPage);
				targetGlyphPage = &glyphPages.back();
				targetGlyphPage->setFileName(textureFileTitle);
			}

			// Add the glyph to the selected page.
			Rect rect = targetGlyphPage->addGlyph(*glyph);

			// Create a descriptor for the current glyph.
			doom3GlyphDescriptor = &self.doom3GlyphDescriptors[characterCode];
			configureGlyphDescriptor(*doom3GlyphDescriptor, *glyph, rect, *targetGlyphPage);

			// Destroy the glyph.
			deleteAndClear(glyph);
		}
		else
		{
			return false;
		}
	}
	else
	{
		// Glyph index has already been used.
		int usedCharacterCode = (*glyphIndexIterator).second;
		doom3GlyphDescriptor = &self.doom3GlyphDescriptors[usedCharacterCode];
	}

	// Write the glyph descriptor to file.
	self.descriptorFile.writeGlyphDescriptor(*doom3GlyphDescriptor);

	return true;
}

//--------------------------------------------------------------------------------------
// Description:	Configure the specified glyph descriptor based on the rectangle and
//				glyph page.
// Parameters:	The glyph descriptor to configure,
//				The glyph,
//				The rectangle of the glyph,
//				The page on which the glyph has been added.
// Returns:		None.
//--------------------------------------------------------------------------------------
void FontExporter::configureGlyphDescriptor(
	Doom3GlyphDescriptor& descriptor,
	const Glyph& glyph,
	const Rect& glyphRect,
	const GlyphPage& glyphPage)
{
	int glyphRectX2 = glyphRect.x + glyphRect.w;
	int glyphRectY2 = glyphRect.y + glyphRect.h;

	descriptor.imageWidth = glyphRect.w;
	descriptor.imageHeight = glyphRect.h;
	descriptor.s = float(glyphRect.x) / glyphPage.getWidth();
	descriptor.s2 = float(glyphRectX2) / glyphPage.getWidth();
	descriptor.t = float(glyphRect.y) / glyphPage.getHeight();
	descriptor.t2 = float(glyphRectY2) / glyphPage.getHeight();
	descriptor.top = glyph.getBitmapTop();

	descriptor.glyph = 0;				// Always zero (it's a pointer that D3 fills in).
	descriptor.pitch = glyphRect.w;		// unused?
	descriptor.height = glyphRect.h;	// unused?
	descriptor.bottom = 0;				// unused?

	descriptor.xSkip = glyph.getXAdvance();

	// Corrections for D3 x offset rendering limitation.
	int bitmapLeft = glyph.getBitmapLeft();
	switch (self.xOffsetFixType)
	{
		case Glyph::XOffsetFixType_None:
		{
			// No fix; no change to xSkip.
			// Encode the BitmapLeft value into the structure.
			descriptor.pitch = bitmapLeft;
		}
		break;

		case Glyph::XOffsetFixType_Glyph:
		{
			// Glyph fix; no change to xSkip.
		}
		break;

		case Glyph::XOffsetFixType_Font:
		{
			// Font-wide fix; change xSkip if bitmapLeft is negative.
			if (bitmapLeft < 0)
			{
				descriptor.xSkip += std::abs(bitmapLeft);
			}

			// Encode the x padding offset into the structure.
			descriptor.pitch = self.font->getXOffsetPadding();
		}
		break;
	}

	// Always tga, even if dds is being output.
	descriptor.shaderName = "fonts/" + glyphPage.getFileName() + ".tga";
}
