//--------------------------------------------------------------------------------------
// File:		glyph.cpp.
// Namespace:	Global.
// Description:	A wrapper for a FreeType glyph.
// Author:		Grant Davies.
// Platform:	ALL.
// 
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Includes.
//--------------------------------------------------------------------------------------

#ifndef __GLYPH__
#include "glyph.h"
#endif //__GLYPH__

#include <assert.h> 

#include <ft2build.h>
#include FT_FREETYPE_H

#ifndef __CORE__
#include "Core.h"
#endif //__CORE__

#ifndef __SURFACEEXTENSIONS__
#include "SurfaceExtensions.h"
#endif //__SURFACEEXTENSIONS__


//--------------------------------------------------------------------------------------
// Constants.
//--------------------------------------------------------------------------------------

#define CLASS_NAME Glyph


//--------------------------------------------------------------------------------------
// Function Definitions.
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Description:	Constructor.
// Parameters:	The type face to use to load this glyph,
//				The index of the glyph to load,
//				[optional] The type of fix that should be applied to this glyph 
//				(default = XOffsetFixType_None),
//				[optional] The amount of blank space that should be added to the left 
//				size of this glyph (plus the positive/negative bitmap left value)
//				(default = 0).
// Returns:		None.
//--------------------------------------------------------------------------------------
Glyph::Glyph(
	const FT_Face& typeFace,
	int glyphIndex,
	XOffsetFixType xOffsetFixType,
	int xPadding)
:
	glyph(0),
	surface(0),
	width(0),
	height(0)
{
	// Load the character.
	self.lastErrorValue = FT_Load_Glyph(typeFace, glyphIndex, FT_LOAD_RENDER); 
	if (isStatusOk())
	{
		// Return the glyph.
		self.glyph = typeFace->glyph;

		// Store the initial width and height.
		width = self.glyph->bitmap.width;
		height = self.glyph->bitmap.rows;

		// Ensure the glyph format is a raster bitmap.
		assert(FT_GLYPH_FORMAT_BITMAP == self.glyph->format);

		// Convert the pixel data to a surface.
		FT_Pixel_Mode pixelMode = FT_Pixel_Mode(self.glyph->bitmap.pixel_mode);

		// Only support gray bitmaps for now.
		assert(FT_PIXEL_MODE_GRAY == pixelMode);

		// Create a surface from the glyph.
		createSurfaceFromGlyph(xOffsetFixType, xPadding);
	}
}

//--------------------------------------------------------------------------------------
// Description:	Destructor.
// Parameters:	None.
// Returns:		None.
//--------------------------------------------------------------------------------------
Glyph::~Glyph()
{
	// Free the surface.
	deleteAndClear(self.surface);
}

//--------------------------------------------------------------------------------------
// Description:	Create a surface from this glyph.
// Parameters:	The type of fix being applied to this glyph,
//				[optional] The amount of blank space that should be added to the left 
//				side of this glyph surface (plus the positve/negative bitmap left 
//				value) (default = 0).
// Returns:		None.
//--------------------------------------------------------------------------------------
void Glyph::createSurfaceFromGlyph(XOffsetFixType xOffsetFixType, int xPadding)
{
	// Create a new surface.
	int glyphWidth = self.getWidth();
	int glyphHeight = self.getHeight();

	if (0 == glyphWidth || 0 == glyphHeight)
	{
		return;
	}

	// The surface must be padded in the x direction to account for the way in which
	// Doom 3 renders.  D3 has no x offset value to use during rendering so it
	// must be in-built into the font.
	int bitmapLeft = self.getBitmapLeft();

	switch (xOffsetFixType)
	{
		case XOffsetFixType_None:
		{
			// No fix at all.  Do not modify the surface width.
			xPadding = 0;
		}
		break;

		case XOffsetFixType_Glyph:
		{
			// Fix the incompatible glyphs only, not the entire font.
			// Pad the glyph out by a non-negative value.
			if (bitmapLeft < 0)
			{
				bitmapLeft = 0;
			}

			xPadding = bitmapLeft;
		}
		break;

		case XOffsetFixType_Font:
		{
			// Fix the entire font.
			// Pad the glyph out by a positive or negative value plus the
			// value that has been passed in.
			xPadding += bitmapLeft;
		}
		break;
	}

	self.surface = createSurface(glyphWidth + xPadding, glyphHeight);

	// Get access to the surface pixels.
	byte* surfacePixels = static_cast<byte*>(self.surface->GetData());

	// Grayscale bitmap is an 8-bit bitmap using 1 byte per pixel.
	const byte* imageBuffer = static_cast<const byte*>(self.glyph->bitmap.buffer);

	for (int y = 0; y < glyphHeight; y++)
	{
		// Get access to the current line data.
		const byte* imageBufferLine = &imageBuffer[y * self.glyph->bitmap.pitch];
		byte* surfaceLinePixelsBytes = &surfacePixels[y * getSurfacePitch(surface)];

		Colour* surfaceLinePixels = (Colour*)surfaceLinePixelsBytes;

		for (int x = 0; x < glyphWidth; x++)
		{
			// Get the pixel colour index of the current pixel.
			int pixelColourIndex = imageBufferLine[x];

			// Store the pixel into the surface.
			assert(pixelColourIndex < 256);
			assert(pixelColourIndex >= 0);

			// Get the pixel to write to.
			Colour& pixel = surfaceLinePixels[xPadding + x];

			// Stored as an alpha value.
			pixel.red = 0xff;
			pixel.green = 0xff;
			pixel.blue = 0xff;
			pixel.alpha = pixelColourIndex;
		}
	}

	// Store the new size of this glyph.
	width = self.surface->Width();
	height = self.surface->Height();
}

//--------------------------------------------------------------------------------------
// Description:	Blit the specified bitmap to the specified surface.
// Parameters:	The surface to draw into,
//				The x position to draw the bitmap at,
//				The y position to draw the bitmap at.
// Returns:		None.
//--------------------------------------------------------------------------------------
void Glyph::blitToSurface(ilImage* surface, int xPosition, int yPosition) const
{
	// Define a destination rect to draw to.
	Rect dstRect;
	dstRect.x = xPosition;
	dstRect.y = yPosition;
	dstRect.w = self.getWidth();
	dstRect.h = self.getHeight();

	if (0 == dstRect.w || 0 == dstRect.h)
	{
		return;
	}

	FT_Pixel_Mode pixelMode = FT_Pixel_Mode(self.glyph->bitmap.pixel_mode);
	assert(FT_PIXEL_MODE_GRAY == pixelMode);		// Only support gray bitmaps for now.

	// Blit the surface.
	int resultCode = blitSurface(self.surface, 0, surface, &dstRect);
	assert(0 == resultCode);
}

//--------------------------------------------------------------------------------------
// Description:	Get the x advance (or x skip) value.
// Parameters:	None.
// Returns:		The x advance value.
//--------------------------------------------------------------------------------------
float Glyph::getXAdvance() const
{
	return self.glyph->advance.x / 64.0f;
}

//--------------------------------------------------------------------------------------
// Description:	Get the y advance (or y skip) value.
// Parameters:	None.
// Returns:		The y advance value.
// Notes:		This is usually (if not always) zero.
//--------------------------------------------------------------------------------------
float Glyph::getYAdvance() const
{
	return self.glyph->advance.y / 64.0f;
}

//--------------------------------------------------------------------------------------
// Description:	Get the width of this glyph.
// Parameters:	None.
// Returns:		The width of this glyph in pixels.
//--------------------------------------------------------------------------------------
int Glyph::getWidth() const
{
	return width;
}

//--------------------------------------------------------------------------------------
// Description:	Get the height of this glyph.
// Parameters:	None.
// Returns:		The height of this glyph in pixels.
//--------------------------------------------------------------------------------------
int Glyph::getHeight() const
{
	return height;
}

//--------------------------------------------------------------------------------------
// Description:	Write this glyph to an image file.
// Parameters:	The file to write this glyph as.
// Returns:		None.
//--------------------------------------------------------------------------------------
void Glyph::writeToFile(std::string fileName)
{
	writeSurfaceToFile(self.surface, fileName);
}

//--------------------------------------------------------------------------------------
// Description:	Get the left indentation of the bitmap.
// Parameters:	None.
// Returns:		The left indentation of the bitmap.
//--------------------------------------------------------------------------------------
int Glyph::getBitmapLeft() const
{
	return self.glyph->bitmap_left;
}

//--------------------------------------------------------------------------------------
// Description:	Get the top indentation of the bitmap.
// Parameters:	None.
// Returns:		The top indentation of the bitmap.
//--------------------------------------------------------------------------------------
int Glyph::getBitmapTop() const
{
	return self.glyph->bitmap_top;
}

//--------------------------------------------------------------------------------------
// Description:	Get a rectangle specifying the bounds of this glyph.
// Parameters:	None.
// Returns:		A rectangle specifying the bounds of this glyph.
//--------------------------------------------------------------------------------------
Rect Glyph::getRect() const
{
	Rect rect;
	rect.w = self.getWidth();
	rect.h = self.getHeight();
	return rect;
}
