//--------------------------------------------------------------------------------------
// File:		GlyphPage.cpp.
// Namespace:	Global.
// Description:	
// Author:		Grant Davies.
// Platform:	ALL.
// 
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Includes.
//--------------------------------------------------------------------------------------

#ifndef __GLYPHPAGE__
#include "GlyphPage.h"
#endif //__GLYPHPAGE__

#include <iostream>
#include <assert.h>

#ifndef __CORE__
#include "Core.h"
#endif //__CORE__

#ifndef __SURFACEEXTENSIONS__
#include "SurfaceExtensions.h"
#endif //__SURFACEEXTENSIONS__


//--------------------------------------------------------------------------------------
// Constants.
//--------------------------------------------------------------------------------------

#define CLASS_NAME GlyphPage


//--------------------------------------------------------------------------------------
// Function Definitions.
//--------------------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Description:	
// Parameters:	
// Returns:		
//--------------------------------------------------------------------------------------
void outputRect(std::ostream& outputStream, Rect& rect)
{
	outputStream << "(x " << rect.x << ", y " << rect.y << ", w " << rect.w << ", h " << rect.h << ")";
}

//--------------------------------------------------------------------------------------
// Description:	Constructor.
// Parameters:	None.
// Returns:		None.
//--------------------------------------------------------------------------------------
GlyphPage::GlyphPage() :
	surface(0)
{
	// Add the entire space to the free stack.
	Rect freeRect;
	freeRect.x = 0;
	freeRect.y = 0;
	freeRect.w = 256;
	freeRect.h = 256;
	self.freeRects.push_back(freeRect);

	// Create a bitmap surface representing the page.
	self.surface = createSurface(freeRect.w, freeRect.h);
}

//--------------------------------------------------------------------------------------
// Description:	Get the index of the area that best fits the specified glyph.
// Parameters:	The glyph to fit in this page.
// Returns:		The index of the area that best fits the specified glyph, or -1 if this
//				page cannot accomodate the specified glyph.
//--------------------------------------------------------------------------------------
int GlyphPage::getIndexOfBestFitArea(const Glyph& glyph) const
{
	int bestFitIndex = -1;
	int bestFitArea = 0x0fffffff;

	Rect glyphRect = getPaddedGlyphRect(glyph);

	// Attempt to find the smallest free space for this glyph.
	for	(int freeRectIndex = 0; freeRectIndex < self.freeRects.size(); freeRectIndex++)
	{
		// Get the current free space rect.
		const Rect& freeSpaceRect = self.freeRects.at(freeRectIndex);

		// Check if this glyph can fit in this rect.
		if (freeSpaceRect.canAccomodateRect(glyphRect) && freeSpaceRect.getArea() < bestFitArea)
		{
			// Store the best fit index.
			bestFitIndex = freeRectIndex;
			bestFitArea = freeSpaceRect.getArea();
		}
	}

	// Return the best fit index.
	return bestFitIndex;
}

//--------------------------------------------------------------------------------------
// Description:	Determine whether this page can accomodate the specified glyph or not.
// Parameters:	The glyph to check.
// Returns:		true if this page can accomodate the specified glyph; false if not.
//--------------------------------------------------------------------------------------
bool GlyphPage::canAccomodateGlyph(const Glyph& glyph) const
{
	return (-1 != self.getIndexOfBestFitArea(glyph));
}

//--------------------------------------------------------------------------------------
// Description:	Add the specified glyph to this page.
// Parameters:	The glyph to add to this page.
// Returns:		A rectangle specifying the position where the glyph was added to this 
//				page.
//--------------------------------------------------------------------------------------
Rect GlyphPage::addGlyph(const Glyph& glyph)
{
	Rect addedRect;

	// The index of the best fit area.
	int bestFitIndex = self.getIndexOfBestFitArea(glyph);

	// If the index is valid, add it.
	if (-1 != bestFitIndex)
	{
		// Get the original area.
		Rect originalRect = self.freeRects.at(bestFitIndex);

		Rect glyphRect = getPaddedGlyphRect(glyph);

		// Add the glyph to the surface.
		glyph.blitToSurface(self.surface, originalRect.x, originalRect.y);

		// Remove the original.
		std::vector<Rect>::iterator bestFitRectIterator = self.freeRects.begin();
		std::advance(bestFitRectIterator, bestFitIndex);
		self.freeRects.erase(bestFitRectIterator);

		// Divide the area into three.
		Rect rectTop;
		rectTop.x = originalRect.x + glyphRect.w;
		rectTop.y = originalRect.y;
		rectTop.w = originalRect.w - glyphRect.w;
		rectTop.h = glyphRect.h;
		if (rectTop.w > 0 && rectTop.h > 0)
		{
			self.freeRects.push_back(rectTop);
		}

		Rect rectLeft;
		rectLeft.x = originalRect.x;
		rectLeft.y = originalRect.y + glyphRect.h;
		rectLeft.w = glyphRect.w;
		rectLeft.h = originalRect.h - glyphRect.h;
		if (rectLeft.w > 0 && rectTop.h > 0)
		{
			self.freeRects.push_back(rectLeft);
		}

		Rect rectRight;
		rectRight.x = originalRect.x + glyphRect.w;
		rectRight.y = originalRect.y + glyphRect.h;
		rectRight.w = originalRect.w - glyphRect.w;
		rectRight.h = originalRect.h - glyphRect.h;
		if (rectRight.w > 0 && rectRight.h > 0)
		{
			self.freeRects.push_back(rectRight);
		}

		// Set the rectangle that the glyph was added as.
		glyphRect = glyph.getRect();
		addedRect = Rect(originalRect.x, originalRect.y, glyphRect.w, glyphRect.h);
	}

	return addedRect;
}

//--------------------------------------------------------------------------------------
// Description:	Blit the specified bitmap to the screen.
// Parameters:	The surface to draw into,
//				The x position to draw the bitmap at,
//				The y position to draw the bitmap at.
// Returns:		None.
//--------------------------------------------------------------------------------------
void GlyphPage::blitToSurface(ilImage* surface, int xPosition, int yPosition) const
{
	// Define a destination rect to draw to.
	Rect dstRect(xPosition, yPosition, 0, 0);

	// Blit the surface.
	int resultCode = blitSurface(self.surface, 0, surface, &dstRect);
	assert(0 == resultCode);
}

//--------------------------------------------------------------------------------------
// Description:	Write this glyph page to disk.
// Parameters:	[optional] The name of the file to write this glyph page as (default = 
//				use current file name).
// Returns:		None.
//--------------------------------------------------------------------------------------
void GlyphPage::writeToFile(std::string fileName)
{
	// If the file name is empty, use the current file name.
	if (fileName.empty())
	{
		fileName = self.fileName;
	}

	// Write the image to the specified file.
	writeSurfaceToFile(self.surface, fileName);
}

//--------------------------------------------------------------------------------------
// Description:	Get the width of this page.
// Parameters:	None.
// Returns:		The width of this page.
//--------------------------------------------------------------------------------------
int GlyphPage::getWidth() const
{
	return self.surface->Width();
}

//--------------------------------------------------------------------------------------
// Description:	Get the height of this page.
// Parameters:	None.
// Returns:		The height of this page.
//--------------------------------------------------------------------------------------
int GlyphPage::getHeight() const
{
	return self.surface->Height();
}

//--------------------------------------------------------------------------------------
// Description:	Set the file name of this page.
// Parameters:	The file name of this page.
// Returns:		None.
//--------------------------------------------------------------------------------------
void GlyphPage::setFileName(std::string fileName)
{
	self.fileName = fileName;
}

//--------------------------------------------------------------------------------------
// Description:	Get the file name of this page.
// Parameters:	None.
// Returns:		The file name of this page.
//--------------------------------------------------------------------------------------
std::string GlyphPage::getFileName() const
{
	return self.fileName;
}

//--------------------------------------------------------------------------------------
// Description:	Get the glyph rectangle, accounting for padding around the edge.
// Parameters:	None.
// Returns:		The padded out glyph rectangle.
//--------------------------------------------------------------------------------------
Rect GlyphPage::getPaddedGlyphRect(const Glyph& glyph) const
{
	// Because of the way Doom 3 indexes the texture page (using floating point 
	// numbers), it is unable to reference exact pixels - there is error.  How much 
	// error, I don't know.
	// For this reason, the rectangle needs to be padded out.
	// [GD 28/4/2005] Changed from 5 pixels to a power of 2 boundary.
	Rect glyphRect = glyph.getRect();
	glyphRect.increaseDimensionsToPowerOf2(5, 5);
	return glyphRect;
}
